<?php

/**
 * CoreCompatibility class
 *
 * @package  WooCommerce Custom Price/Compatibility
 * @since  1.5.0
 */
namespace CPWFreeVendor\WPDesk\Library\CustomPrice\Compatibility\Core;

use WC_Product;
/**
 * Functions for WC core back-compatibility.
 *
 * @class  CoreCompatibility
 * @since 1.0.0
 */
class CoreCompatibility
{
    /**
     * Cache 'gte' comparison results.
     *
     * @var array
     */
    private static $is_wc_version_gte = array();
    /**
     * Cache 'gt' comparison results.
     *
     * @var array
     */
    private static $is_wc_version_gt = array();
    /**
     * Helper method to get the version of the currently installed WooCommerce.
     *
     * @since 1.0.0
     *
     * @return string
     */
    private static function get_wc_version()
    {
        return \defined('WC_VERSION') && \WC_VERSION ? \WC_VERSION : null;
    }
    /**
     * Returns true if the installed version of WooCommerce is greater than or equal to $version.
     *
     * @since 1.0.0
     *
     * @param  string $version the version to compare
     * @return boolean true if the installed version of WooCommerce is >= $version
     */
    public static function is_wc_version_gte($version)
    {
        if (!isset(self::$is_wc_version_gte[$version])) {
            self::$is_wc_version_gte[$version] = self::get_wc_version() && \version_compare(self::get_wc_version(), $version, '>=');
        }
        return self::$is_wc_version_gte[$version];
    }
    /**
     * Back-compat wrapper for 'get_parent_id'.
     *
     * @param  WC_Product $product
     * @return mixed
     * @since 1.0.0
     */
    public static function get_parent_id($product)
    {
        if (self::is_wc_version_gte('3.0.0')) {
            return $product->get_parent_id();
        } else {
            return $product->is_type('variation') ? \absint($product->get_id) : 0;
        }
    }
    /**
     * Back-compat wrapper for 'get_id'.
     *
     * @since 1.0.0
     *
     * @param  WC_Product $product
     * @return mixed
     */
    public static function get_id($product)
    {
        if (\is_object($product)) {
            if (self::is_wc_version_gte('3.0.0')) {
                $product_id = $product->get_id();
            } else {
                $product_id = $product->is_type('variation') ? $product->variation_id : $product->id;
            }
        } else {
            $product_id = $product;
        }
        return \absint($product_id);
    }
    /**
     * Back-compat wrapper for getting CRUD object props directly.
     *
     * @since 1.0.0
     *
     * @param  object $obj
     * @param  string $prop
     * @param  string $context
     * @return mixed
     */
    public static function get_prop($obj, $name, $context = 'view')
    {
        if (self::is_wc_version_gte('3.0.0')) {
            $get_fn = 'get_' . $name;
            return \is_callable(array($obj, $get_fn)) ? $obj->{$get_fn}($context) : $obj->get_meta('_' . $name, \true);
        } else {
            if ('status' === $name) {
                $value = isset($obj->post->post_status) ? $obj->post->post_status : null;
            } elseif ('short_description' === $name) {
                $value = isset($obj->post->post_excerpt) ? $obj->post->post_excerpt : null;
            } else {
                $value = $obj->{$name};
            }
            return $value;
        }
    }
    /**
     * Back-compat wrapper for setting CRUD object props directly.
     *
     * @since 1.0.0
     *
     * @param  WC_Product $product
     * @param  string     $prop
     * @param  mixed      $value
     * @return void
     */
    public static function set_prop($obj, $name, $value)
    {
        if (self::is_wc_version_gte('3.0.0')) {
            $set_fn = 'set_' . $name;
            if (\is_callable(array($obj, $set_fn))) {
                $obj->{$set_fn}($value);
            } else {
                $obj->add_meta_data('_' . $name, $value, \true);
            }
        } else {
            $obj->{$name} = $value;
        }
    }
    /**
     * Get the "From" text
     *
     * @param int $product
     * @return  string
     * @since 1.0.0
     */
    public static function get_price_html_from_text($product)
    {
        return self::is_wc_version_gte('3.0.0') ? \wc_get_price_html_from_text() : $product->get_price_html_from_text();
    }
}
